# include <stdio.h>
# include <stdlib.h>
# include <asl.h>
# include <math.h>
# include <mpi.h>
# include "sc24.h"

# define Nc_max (L_max_SC+2)*(L_max_SC+2)

int Nc;

int x_cell[Nc_max+1];
int y_cell[Nc_max+1];

int n_cell[L_max_SC+2][L_max_SC+2];
int nn_cell[Nc_max+1][5];
int nnn_cell[Nc_max+1][5];

int island[Nc_max+1];
int n_island[Nc_max+1];
int n_island_rev[Nc_max+1];

double Tree[Nc_max+1];
double Fire[Nc_max+1];
double Empty[Nc_max+1];

double Tree0[Nc_max+1];
double Fire0[Nc_max+1];
double Empty0[Nc_max+1];

void construct_square_lattice();
void construct_island(int i_prob);
void initialization(int i_prob);

void sim(int n_step);

asl_random_t rng;
double random_num[Nc_max];

int iflag=0;
/*int iflag=1;*/

int main(int argc, char *argv[]){

  int myid,n_procs;

  unsigned int seed0=12345;
  unsigned int seed[80];  

  MPI_Init(&argc, &argv);
  SC_input();
  
  MPI_Comm_size(MPI_COMM_WORLD, &n_procs);
  MPI_Comm_rank(MPI_COMM_WORLD, &myid);

  if (myid == 0){
    srand(seed0);
    for (int n=0; n < n_procs; n++){
      seed[n]=(unsigned int) rand();
    };
  };
  MPI_Bcast(&seed,n_procs,MPI_INTEGER,0,MPI_COMM_WORLD);
   
  /* ASL Library Initialization */
  asl_library_initialize();
   
  /* Random Generator Preparation */
  asl_random_parallel_create(&rng, ASL_RANDOMMETHOD_MT19937_64,MPI_COMM_WORLD);
  asl_random_initialize(rng,64,&seed[myid]);
  asl_random_distribute_uniform(rng);

  construct_square_lattice();

  for (int i_prob=0; i_prob < N_prob_SC; i_prob++){

    if (myid ==0 && iflag ==1 ){
      printf("#i_prob= %d %d \n",i_prob,Nc_SC[i_prob]);
    }

    construct_island(i_prob);

    initialization(i_prob);

    sim(tw_SC);

    double sum_Tree;
    double sum_Fire;
    double sum_Empty;

    sum_Tree=0.0;
    sum_Fire=0.0;
    sum_Empty=0.0;

    for (int n=1; n <= Nc_SC[i_prob]; n++){
      int m=n_island[n];
      sum_Tree=sum_Tree+Tree0[m];
      sum_Fire=sum_Fire+Fire0[m];
      sum_Empty=sum_Empty+Empty0[m];
    }

    double cut_tree_total_limit=sum_Fire*R_SC;

    MPI_Barrier(MPI_COMM_WORLD);
    /* Generate new random numbers */
    asl_random_generate_d(rng, Nc_max, random_num);

    double weight[Nc_max+1];

    double amp_random=0.5;

    for (int n=1; n<=Nc_SC[i_prob]; n++){
      if (Fire0[n] > 0.0){
	weight[n]=0.5+amp_random*(2.0*random_num[n-1]-1.0);
      }
      else
	{weight[n]=0.0;}
    }

    double sfactor=1.0;
    int iflag_s=1;
    double cut_tree_total=0.0;
    
    while (iflag_s ==1){
    
      double total_cut=0.0;
      for (int n=1; n<=Nc_SC[i_prob]; n++){
	int m=n_island[n];
	total_cut=total_cut+weight[n]*Tree0[m];
      }
      double amp=fmin(sfactor*cut_tree_total_limit/total_cut,1.0);

      cut_tree_total=0.0;
      for (int n=1; n<=Nc_SC[i_prob]; n++){
	int m=n_island[n];
	double d_cut_tree=fmin(amp*weight[n],1.0)*Tree0[m];
	cut_tree_total=cut_tree_total+d_cut_tree;
	Delta_T_SC[i_prob][n-1]=d_cut_tree;
	Tree0[m] =Tree0[m]     -d_cut_tree;
	Empty0[m]=Empty0[m]    +d_cut_tree;
      }

      if (cut_tree_total >cut_tree_total_limit){
	sfactor=sfactor*0.99;
      }
      else
	{iflag_s=0;}
    }
	
    if (myid ==0 && iflag == 1){
      printf("#cut_tree_total,limit= %f %f\n",			\
	     cut_tree_total,cut_tree_total_limit);
    }
    
    sim(tc_SC-tw_SC);

    double survived_Tree=0.0;
    for (int n=1; n <= Nc_SC[i_prob]; n++){
      int m=n_island[n];
      survived_Tree=survived_Tree+Tree0[m];
    }

    double *survived_Tree_all;
    survived_Tree_all=(double *)malloc(n_procs*sizeof(double));

    MPI_Gather(&survived_Tree, 1, MPI_DOUBLE, survived_Tree_all, 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);

    int n_max;  

    if (myid==0){

      double survived_Tree_max;

      survived_Tree_max=survived_Tree_all[0];
      n_max=0;
      for (int n=0; n < n_procs; n++){
	if (iflag > 0){
	  printf("#survived_Tree_all[%d]=%lf\n",n,survived_Tree_all[n]);
	}
	if (survived_Tree_max < survived_Tree_all[n]){
	  survived_Tree_max=survived_Tree_all[n];
	  n_max=n;
	}
      }
    
      if (iflag > 0){
	printf("#survived_Tree_max,n_max= %lf %d\n",survived_Tree_max,n_max);
      }
      
    }

    MPI_Bcast(&n_max,1,MPI_INT,0,MPI_COMM_WORLD);

    MPI_Request requests[2];

    if (n_max > 0){     

      int itag=0;

      MPI_Barrier(MPI_COMM_WORLD);
      if (myid == n_max){
	MPI_Isend(&Delta_T_SC[i_prob][0],Nc_SC[i_prob],MPI_DOUBLE,0,itag,\
		  MPI_COMM_WORLD,&requests[0]);
      }else{
	if (myid ==0){
	  MPI_Irecv(&Delta_T_SC[i_prob][0],Nc_SC[i_prob],MPI_DOUBLE,n_max,itag,\
		    MPI_COMM_WORLD,&requests[1]);
	}
      }
    }

    MPI_Barrier(MPI_COMM_WORLD);
    if (iflag ==0){
      SC_output(i_prob);
    }

  }

  /* Random Generator Finalization */
  MPI_Barrier(MPI_COMM_WORLD);
  asl_random_destroy(rng);

  /* ASL Library Finalization */
  asl_library_finalize();

#ifdef MPI_VERSION
  MPI_Finalize();
#endif
  
}

void  construct_island(int i_prob){

  for (int n=1; n <=Nc_max; n++){
    island[n]=0;
  }

  for (int n=1; n <=Nc_SC[i_prob]; n++){
    int x=x_island_SC[i_prob][n-1];
    int y=y_island_SC[i_prob][n-1];
    int m=n_cell[x][y];
    /* this cell belongs the island*/
    island[m]=1;
    n_island[n]=m;
    n_island_rev[m]=n;
  }
}


void construct_square_lattice(){ 

  int i,j,n;

  n=0;
  for (i=1; i<= L_max_SC; i++){
    for (j=1; j <= L_max_SC; j++){
      n++;
      x_cell[n]=i;
      y_cell[n]=j;
      n_cell[i][j]=n;
    }
  }
  Nc=n;

  j=0;
  for (i=0; i <= L_max_SC+1; i++){
    n++;
    x_cell[n]=i;
    y_cell[n]=j;
    n_cell[i][j]=n;    
  }

  i=0;
  for (j=1; j <= L_max_SC; j++){
    n++;
    x_cell[n]=i;
    y_cell[n]=j;
    n_cell[i][j]=n;
  }

  i=L_max_SC+1;
  for (j=1; j <= L_max_SC; j++){
    n++;
    x_cell[n]=i;
    y_cell[n]=j;
    n_cell[i][j]=n;    
  }

  j=L_max_SC+1;
  for (i=0; i <= L_max_SC+1; i++){
    n++;
    x_cell[n]=i;
    y_cell[n]=j;
    n_cell[i][j]=n;
  }

  for (n=1; n <= Nc; n++){
    nn_cell[n][1]=n_cell[x_cell[n]+1][y_cell[n]  ];
    nn_cell[n][2]=n_cell[x_cell[n]  ][y_cell[n]+1];
    nn_cell[n][3]=n_cell[x_cell[n]-1][y_cell[n]  ];
    nn_cell[n][4]=n_cell[x_cell[n]  ][y_cell[n]-1];
  }

  for (n=1; n <= Nc; n++){
    nnn_cell[n][1]=n_cell[x_cell[n]+1][y_cell[n]+1];
    nnn_cell[n][2]=n_cell[x_cell[n]-1][y_cell[n]+1];
    nnn_cell[n][3]=n_cell[x_cell[n]-1][y_cell[n]-1];
    nnn_cell[n][4]=n_cell[x_cell[n]+1][y_cell[n]-1];
  }
  
}


void initialization(int i_prob){

  for (int n=1; n <= Nc_max; n++){
    
    Tree0[n]=0.0;
    Fire0[n]=0.0;
    Empty0[n]=1.0;
    
  }

  for (int n=1; n <= Nc_SC[i_prob]; n++){

    int x=x_island_SC[i_prob][n-1];
    int y=y_island_SC[i_prob][n-1];
    int m=n_cell[x][y];

    Tree0[m]=T_SC[i_prob][n-1];
    Fire0[m]=F_SC[i_prob][n-1];
    Empty0[m]=1.0-Tree0[m]-Fire0[m];
    
  }

}
 

void sim(int n_step){

  double contact;

  for (int istep=1; istep <= n_step; istep++){

    for (int n=1; n <= Nc; n++){
      
      contact=beta1_SC*Fire0[n]+			\
	+beta2_SC*(Fire0[nn_cell[n][1]]+		\
		   Fire0[nn_cell[n][2]]+		\
		   Fire0[nn_cell[n][3]]+		\
		   Fire0[nn_cell[n][4]])		\
	+beta3_SC*(Fire0[nnn_cell[n][1]]+		\
		   Fire0[nnn_cell[n][2]]+		\
		   Fire0[nnn_cell[n][3]]+		\
		   Fire0[nnn_cell[n][4]]);      
      if (contact < hc_SC){
	contact=0.0;}
      else
	{contact=tanh(A_SC*(contact-hc_SC));}

      if (island[n]==1){
	Tree[n] =Tree0[n] -contact*Tree0[n];
	Fire[n] =Fire0[n] +contact*Tree0[n]-gamma_SC*Fire0[n];
	Empty[n]=Empty0[n]+gamma_SC*Fire0[n];
      }else
	{
	  Tree[n] =Tree0[n];
	  Fire[n] =Fire0[n];
	  Empty[n]=Empty0[n];
	}
    }


    for (int n=1; n <= Nc; n++){ 
      Tree0[n]=Tree[n];
      Fire0[n]=Fire[n];
      Empty0[n]=Empty[n];
    }

  }
}







